from fastapi import FastAPI, APIRouter, HTTPException, Depends, Request, Header, Response
from fastapi.responses import RedirectResponse, JSONResponse
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
from pathlib import Path
from pydantic import BaseModel, Field, EmailStr
from typing import List, Optional, Dict, Any
import uuid
from datetime import datetime, timezone, timedelta
import hashlib
import secrets
import jwt
import bcrypt
import qrcode
from io import BytesIO
import base64
import re
from user_agents import parse as parse_user_agent

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# MongoDB connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

# JWT Settings
JWT_SECRET = os.environ.get('JWT_SECRET', 'default_secret_key')
JWT_ALGORITHM = "HS256"
JWT_EXPIRATION_HOURS = 24

# Stripe settings
STRIPE_API_KEY = os.environ.get('STRIPE_API_KEY', 'sk_test_emergent')

# Subscription Plans
SUBSCRIPTION_PLANS = {
    "free": {"name": "Free", "price": 0.0, "links_per_month": 3, "features": ["Basic analytics", "Standard support"]},
    "basic": {"name": "Basic", "price": 9.99, "links_per_month": 30, "features": ["Advanced analytics", "Priority support", "QR codes"]},
    "pro": {"name": "Pro", "price": 29.99, "links_per_month": 300, "features": ["Full analytics", "API access", "Password protection", "Custom aliases"]},
    "ultimate": {"name": "Ultimate", "price": 79.99, "links_per_month": 1000, "features": ["Everything in Pro", "Custom domains", "White-label", "Dedicated support"]},
    "enterprise": {"name": "Enterprise", "price": 0.0, "links_per_month": -1, "features": ["Unlimited links", "Custom integrations", "SLA", "Account manager"]}
}

# Create the main app
app = FastAPI(title="Boltlytics API", version="1.0.0")
api_router = APIRouter(prefix="/api")

# ==================== MODELS ====================

class UserCreate(BaseModel):
    email: EmailStr
    password: str
    name: str

class UserLogin(BaseModel):
    email: EmailStr
    password: str

class UserResponse(BaseModel):
    id: str
    email: str
    name: str
    role: str
    subscription_plan: str
    links_created_this_month: int
    created_at: str

class LinkCreate(BaseModel):
    target_url: str
    custom_alias: Optional[str] = None
    password: Optional[str] = None
    expires_at: Optional[str] = None
    location_restrictions: Optional[List[str]] = None
    title: Optional[str] = None

class LinkResponse(BaseModel):
    id: str
    short_code: str
    target_url: str
    title: Optional[str]
    clicks: int
    created_at: str
    expires_at: Optional[str]
    is_password_protected: bool
    location_restrictions: Optional[List[str]]
    qr_code: Optional[str] = None

class LinkUpdate(BaseModel):
    target_url: Optional[str] = None
    title: Optional[str] = None
    password: Optional[str] = None
    expires_at: Optional[str] = None
    location_restrictions: Optional[List[str]] = None

class ClickAnalytics(BaseModel):
    total_clicks: int
    clicks_today: int
    clicks_this_week: int
    clicks_this_month: int
    top_countries: List[Dict[str, Any]]
    top_referrers: List[Dict[str, Any]]
    top_browsers: List[Dict[str, Any]]
    top_devices: List[Dict[str, Any]]
    clicks_by_day: List[Dict[str, Any]]

class APIKeyCreate(BaseModel):
    name: str

class APIKeyResponse(BaseModel):
    id: str
    name: str
    key_prefix: str
    created_at: str
    last_used: Optional[str]
    is_active: bool

class SubscriptionCheckout(BaseModel):
    plan_id: str
    origin_url: str
    payment_method: Optional[str] = "card"

class PasswordVerify(BaseModel):
    password: str

# ==================== AUTH HELPERS ====================

def hash_password(password: str) -> str:
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')

def verify_password(password: str, hashed: str) -> bool:
    return bcrypt.checkpw(password.encode('utf-8'), hashed.encode('utf-8'))

def create_token(user_id: str, email: str, role: str) -> str:
    payload = {
        "user_id": user_id,
        "email": email,
        "role": role,
        "exp": datetime.now(timezone.utc) + timedelta(hours=JWT_EXPIRATION_HOURS)
    }
    return jwt.encode(payload, JWT_SECRET, algorithm=JWT_ALGORITHM)

async def get_current_user(authorization: Optional[str] = Header(None)):
    if not authorization or not authorization.startswith("Bearer "):
        raise HTTPException(status_code=401, detail="Not authenticated")
    
    token = authorization.split(" ")[1]
    try:
        payload = jwt.decode(token, JWT_SECRET, algorithms=[JWT_ALGORITHM])
        user = await db.users.find_one({"id": payload["user_id"]}, {"_id": 0})
        if not user:
            raise HTTPException(status_code=401, detail="User not found")
        return user
    except jwt.ExpiredSignatureError:
        raise HTTPException(status_code=401, detail="Token expired")
    except jwt.InvalidTokenError:
        raise HTTPException(status_code=401, detail="Invalid token")

async def get_admin_user(user: dict = Depends(get_current_user)):
    if user.get("role") != "admin":
        raise HTTPException(status_code=403, detail="Admin access required")
    return user

async def verify_api_key(x_api_key: Optional[str] = Header(None)):
    if not x_api_key:
        raise HTTPException(status_code=401, detail="API key required")
    
    key_hash = hashlib.sha256(x_api_key.encode()).hexdigest()
    api_key = await db.api_keys.find_one({"key_hash": key_hash, "is_active": True}, {"_id": 0})
    
    if not api_key:
        raise HTTPException(status_code=401, detail="Invalid API key")
    
    # Update last used
    await db.api_keys.update_one(
        {"key_hash": key_hash},
        {"$set": {"last_used": datetime.now(timezone.utc).isoformat()}}
    )
    
    user = await db.users.find_one({"id": api_key["user_id"]}, {"_id": 0})
    return user

# ==================== UTILITY FUNCTIONS ====================

def generate_short_code(length: int = 6) -> str:
    chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
    return ''.join(secrets.choice(chars) for _ in range(length))

def generate_qr_code(url: str) -> str:
    qr = qrcode.QRCode(version=1, box_size=10, border=5)
    qr.add_data(url)
    qr.make(fit=True)
    img = qr.make_image(fill_color="black", back_color="white")
    buffer = BytesIO()
    img.save(buffer, format="PNG")
    return base64.b64encode(buffer.getvalue()).decode()

def parse_request_info(request: Request, user_agent_str: str) -> dict:
    ua = parse_user_agent(user_agent_str)
    
    # Get IP (check for forwarded headers)
    ip = request.headers.get("X-Forwarded-For", request.client.host if request.client else "unknown")
    if "," in ip:
        ip = ip.split(",")[0].strip()
    
    return {
        "ip": ip,
        "browser": ua.browser.family,
        "browser_version": ua.browser.version_string,
        "os": ua.os.family,
        "os_version": ua.os.version_string,
        "device": "Mobile" if ua.is_mobile else "Tablet" if ua.is_tablet else "Desktop",
        "is_bot": ua.is_bot
    }

async def check_link_limit(user: dict) -> bool:
    plan = SUBSCRIPTION_PLANS.get(user.get("subscription_plan", "free"), SUBSCRIPTION_PLANS["free"])
    if plan["links_per_month"] == -1:
        return True
    
    # Count links created this month
    start_of_month = datetime.now(timezone.utc).replace(day=1, hour=0, minute=0, second=0, microsecond=0)
    count = await db.links.count_documents({
        "user_id": user["id"],
        "created_at": {"$gte": start_of_month.isoformat()}
    })
    
    return count < plan["links_per_month"]

# ==================== AUTH ENDPOINTS ====================

@api_router.post("/auth/register")
async def register(user_data: UserCreate):
    # Check if email exists
    existing = await db.users.find_one({"email": user_data.email})
    if existing:
        raise HTTPException(status_code=400, detail="Email already registered")
    
    user_id = str(uuid.uuid4())
    user = {
        "id": user_id,
        "email": user_data.email,
        "password": hash_password(user_data.password),
        "name": user_data.name,
        "role": "user",
        "subscription_plan": "free",
        "links_created_this_month": 0,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.users.insert_one(user)
    token = create_token(user_id, user_data.email, "user")
    
    return {
        "token": token,
        "user": {
            "id": user_id,
            "email": user_data.email,
            "name": user_data.name,
            "role": "user",
            "subscription_plan": "free"
        }
    }

@api_router.post("/auth/login")
async def login(credentials: UserLogin):
    user = await db.users.find_one({"email": credentials.email}, {"_id": 0})
    if not user or not verify_password(credentials.password, user["password"]):
        raise HTTPException(status_code=401, detail="Invalid credentials")
    
    token = create_token(user["id"], user["email"], user["role"])
    
    return {
        "token": token,
        "user": {
            "id": user["id"],
            "email": user["email"],
            "name": user["name"],
            "role": user["role"],
            "subscription_plan": user.get("subscription_plan", "free")
        }
    }

@api_router.get("/auth/me")
async def get_me(user: dict = Depends(get_current_user)):
    # Count links this month
    start_of_month = datetime.now(timezone.utc).replace(day=1, hour=0, minute=0, second=0, microsecond=0)
    links_count = await db.links.count_documents({
        "user_id": user["id"],
        "created_at": {"$gte": start_of_month.isoformat()}
    })
    
    plan = SUBSCRIPTION_PLANS.get(user.get("subscription_plan", "free"), SUBSCRIPTION_PLANS["free"])
    
    return {
        "id": user["id"],
        "email": user["email"],
        "name": user["name"],
        "role": user["role"],
        "subscription_plan": user.get("subscription_plan", "free"),
        "links_created_this_month": links_count,
        "links_limit": plan["links_per_month"],
        "created_at": user.get("created_at", "")
    }

# ==================== LINKS ENDPOINTS ====================

@api_router.post("/links")
async def create_link(link_data: LinkCreate, request: Request, user: dict = Depends(get_current_user)):
    # Check link limit
    can_create = await check_link_limit(user)
    if not can_create:
        raise HTTPException(status_code=403, detail="Link limit reached for your plan. Please upgrade.")
    
    # Validate URL
    if not link_data.target_url.startswith(("http://", "https://")):
        link_data.target_url = "https://" + link_data.target_url
    
    # Generate or validate custom alias
    if link_data.custom_alias:
        if not re.match(r'^[a-zA-Z0-9_-]+$', link_data.custom_alias):
            raise HTTPException(status_code=400, detail="Invalid custom alias format")
        existing = await db.links.find_one({"short_code": link_data.custom_alias})
        if existing:
            raise HTTPException(status_code=400, detail="Custom alias already taken")
        short_code = link_data.custom_alias
    else:
        short_code = generate_short_code()
        while await db.links.find_one({"short_code": short_code}):
            short_code = generate_short_code()
    
    link_id = str(uuid.uuid4())
    
    # Generate QR code
    base_url = str(request.base_url).rstrip('/')
    short_url = f"{base_url}/s/{short_code}"
    qr_code = generate_qr_code(short_url)
    
    link = {
        "id": link_id,
        "user_id": user["id"],
        "short_code": short_code,
        "target_url": link_data.target_url,
        "title": link_data.title or link_data.target_url[:50],
        "password": hash_password(link_data.password) if link_data.password else None,
        "expires_at": link_data.expires_at,
        "location_restrictions": link_data.location_restrictions,
        "clicks": 0,
        "qr_code": qr_code,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.links.insert_one(link)
    
    return {
        "id": link_id,
        "short_code": short_code,
        "short_url": short_url,
        "target_url": link_data.target_url,
        "title": link["title"],
        "clicks": 0,
        "created_at": link["created_at"],
        "expires_at": link_data.expires_at,
        "is_password_protected": bool(link_data.password),
        "location_restrictions": link_data.location_restrictions,
        "qr_code": qr_code
    }

@api_router.get("/links")
async def get_links(user: dict = Depends(get_current_user), skip: int = 0, limit: int = 50):
    query = {"user_id": user["id"]}
    links = await db.links.find(query, {"_id": 0, "password": 0}).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    total = await db.links.count_documents(query)
    
    return {
        "links": links,
        "total": total,
        "skip": skip,
        "limit": limit
    }

@api_router.get("/links/{link_id}")
async def get_link(link_id: str, user: dict = Depends(get_current_user)):
    link = await db.links.find_one({"id": link_id, "user_id": user["id"]}, {"_id": 0, "password": 0})
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    return link

@api_router.put("/links/{link_id}")
async def update_link(link_id: str, link_data: LinkUpdate, user: dict = Depends(get_current_user)):
    link = await db.links.find_one({"id": link_id, "user_id": user["id"]})
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    
    update_data = {"updated_at": datetime.now(timezone.utc).isoformat()}
    
    if link_data.target_url:
        update_data["target_url"] = link_data.target_url
    if link_data.title:
        update_data["title"] = link_data.title
    if link_data.password is not None:
        update_data["password"] = hash_password(link_data.password) if link_data.password else None
    if link_data.expires_at is not None:
        update_data["expires_at"] = link_data.expires_at
    if link_data.location_restrictions is not None:
        update_data["location_restrictions"] = link_data.location_restrictions
    
    await db.links.update_one({"id": link_id}, {"$set": update_data})
    
    updated_link = await db.links.find_one({"id": link_id}, {"_id": 0, "password": 0})
    return updated_link

@api_router.delete("/links/{link_id}")
async def delete_link(link_id: str, user: dict = Depends(get_current_user)):
    result = await db.links.delete_one({"id": link_id, "user_id": user["id"]})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="Link not found")
    
    # Also delete associated clicks
    await db.clicks.delete_many({"link_id": link_id})
    
    return {"message": "Link deleted successfully"}

# ==================== ANALYTICS ENDPOINTS ====================

@api_router.get("/links/{link_id}/analytics")
async def get_link_analytics(link_id: str, user: dict = Depends(get_current_user)):
    link = await db.links.find_one({"id": link_id, "user_id": user["id"]}, {"_id": 0})
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    
    now = datetime.now(timezone.utc)
    today_start = now.replace(hour=0, minute=0, second=0, microsecond=0)
    week_start = today_start - timedelta(days=7)
    month_start = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
    
    # Get click counts
    total_clicks = await db.clicks.count_documents({"link_id": link_id})
    clicks_today = await db.clicks.count_documents({"link_id": link_id, "created_at": {"$gte": today_start.isoformat()}})
    clicks_this_week = await db.clicks.count_documents({"link_id": link_id, "created_at": {"$gte": week_start.isoformat()}})
    clicks_this_month = await db.clicks.count_documents({"link_id": link_id, "created_at": {"$gte": month_start.isoformat()}})
    
    # Aggregate by country
    countries_pipeline = [
        {"$match": {"link_id": link_id}},
        {"$group": {"_id": "$country", "count": {"$sum": 1}}},
        {"$sort": {"count": -1}},
        {"$limit": 10}
    ]
    top_countries = await db.clicks.aggregate(countries_pipeline).to_list(10)
    
    # Aggregate by referrer
    referrers_pipeline = [
        {"$match": {"link_id": link_id}},
        {"$group": {"_id": "$referrer", "count": {"$sum": 1}}},
        {"$sort": {"count": -1}},
        {"$limit": 10}
    ]
    top_referrers = await db.clicks.aggregate(referrers_pipeline).to_list(10)
    
    # Aggregate by browser
    browsers_pipeline = [
        {"$match": {"link_id": link_id}},
        {"$group": {"_id": "$browser", "count": {"$sum": 1}}},
        {"$sort": {"count": -1}},
        {"$limit": 5}
    ]
    top_browsers = await db.clicks.aggregate(browsers_pipeline).to_list(5)
    
    # Aggregate by device
    devices_pipeline = [
        {"$match": {"link_id": link_id}},
        {"$group": {"_id": "$device", "count": {"$sum": 1}}},
        {"$sort": {"count": -1}},
        {"$limit": 5}
    ]
    top_devices = await db.clicks.aggregate(devices_pipeline).to_list(5)
    
    # Clicks by day (last 30 days)
    thirty_days_ago = today_start - timedelta(days=30)
    clicks_by_day_pipeline = [
        {"$match": {"link_id": link_id, "created_at": {"$gte": thirty_days_ago.isoformat()}}},
        {"$addFields": {"date": {"$substr": ["$created_at", 0, 10]}}},
        {"$group": {"_id": "$date", "count": {"$sum": 1}}},
        {"$sort": {"_id": 1}}
    ]
    clicks_by_day = await db.clicks.aggregate(clicks_by_day_pipeline).to_list(30)
    
    return {
        "total_clicks": total_clicks,
        "clicks_today": clicks_today,
        "clicks_this_week": clicks_this_week,
        "clicks_this_month": clicks_this_month,
        "top_countries": [{"country": c["_id"] or "Unknown", "count": c["count"]} for c in top_countries],
        "top_referrers": [{"referrer": r["_id"] or "Direct", "count": r["count"]} for r in top_referrers],
        "top_browsers": [{"browser": b["_id"] or "Unknown", "count": b["count"]} for b in top_browsers],
        "top_devices": [{"device": d["_id"] or "Unknown", "count": d["count"]} for d in top_devices],
        "clicks_by_day": [{"date": c["_id"], "count": c["count"]} for c in clicks_by_day]
    }

@api_router.get("/analytics/overview")
async def get_analytics_overview(user: dict = Depends(get_current_user)):
    now = datetime.now(timezone.utc)
    today_start = now.replace(hour=0, minute=0, second=0, microsecond=0)
    week_start = today_start - timedelta(days=7)
    month_start = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
    
    # Get user's link IDs
    user_links = await db.links.find({"user_id": user["id"]}, {"id": 1}).to_list(None)
    link_ids = [l["id"] for l in user_links]
    
    if not link_ids:
        return {
            "total_links": 0,
            "total_clicks": 0,
            "clicks_today": 0,
            "clicks_this_week": 0,
            "clicks_this_month": 0,
            "top_links": [],
            "clicks_by_day": []
        }
    
    # Get click counts
    total_clicks = await db.clicks.count_documents({"link_id": {"$in": link_ids}})
    clicks_today = await db.clicks.count_documents({"link_id": {"$in": link_ids}, "created_at": {"$gte": today_start.isoformat()}})
    clicks_this_week = await db.clicks.count_documents({"link_id": {"$in": link_ids}, "created_at": {"$gte": week_start.isoformat()}})
    clicks_this_month = await db.clicks.count_documents({"link_id": {"$in": link_ids}, "created_at": {"$gte": month_start.isoformat()}})
    
    # Top performing links
    top_links = await db.links.find(
        {"user_id": user["id"]},
        {"_id": 0, "id": 1, "short_code": 1, "title": 1, "clicks": 1}
    ).sort("clicks", -1).limit(5).to_list(5)
    
    # Clicks by day (last 30 days)
    thirty_days_ago = today_start - timedelta(days=30)
    clicks_by_day_pipeline = [
        {"$match": {"link_id": {"$in": link_ids}, "created_at": {"$gte": thirty_days_ago.isoformat()}}},
        {"$addFields": {"date": {"$substr": ["$created_at", 0, 10]}}},
        {"$group": {"_id": "$date", "count": {"$sum": 1}}},
        {"$sort": {"_id": 1}}
    ]
    clicks_by_day = await db.clicks.aggregate(clicks_by_day_pipeline).to_list(30)
    
    return {
        "total_links": len(link_ids),
        "total_clicks": total_clicks,
        "clicks_today": clicks_today,
        "clicks_this_week": clicks_this_week,
        "clicks_this_month": clicks_this_month,
        "top_links": top_links,
        "clicks_by_day": [{"date": c["_id"], "count": c["count"]} for c in clicks_by_day]
    }

# ==================== API KEY ENDPOINTS ====================

@api_router.post("/api-keys")
async def create_api_key(key_data: APIKeyCreate, user: dict = Depends(get_current_user)):
    # Check if user has pro or above plan
    if user.get("subscription_plan") not in ["pro", "ultimate", "enterprise"]:
        raise HTTPException(status_code=403, detail="API access requires Pro plan or above")
    
    # Generate API key
    api_key = f"sk_{secrets.token_urlsafe(32)}"
    key_hash = hashlib.sha256(api_key.encode()).hexdigest()
    
    key_doc = {
        "id": str(uuid.uuid4()),
        "user_id": user["id"],
        "name": key_data.name,
        "key_hash": key_hash,
        "key_prefix": api_key[:12],
        "created_at": datetime.now(timezone.utc).isoformat(),
        "last_used": None,
        "is_active": True
    }
    
    await db.api_keys.insert_one(key_doc)
    
    # Return the full key only once
    return {
        "id": key_doc["id"],
        "name": key_doc["name"],
        "key": api_key,
        "key_prefix": key_doc["key_prefix"],
        "created_at": key_doc["created_at"],
        "message": "Save this key securely. It won't be shown again."
    }

@api_router.get("/api-keys")
async def get_api_keys(user: dict = Depends(get_current_user)):
    keys = await db.api_keys.find(
        {"user_id": user["id"]},
        {"_id": 0, "key_hash": 0}
    ).to_list(None)
    return {"keys": keys}

@api_router.delete("/api-keys/{key_id}")
async def delete_api_key(key_id: str, user: dict = Depends(get_current_user)):
    result = await db.api_keys.delete_one({"id": key_id, "user_id": user["id"]})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="API key not found")
    return {"message": "API key deleted successfully"}

# ==================== PUBLIC API ENDPOINTS ====================

@api_router.post("/v1/links")
async def api_create_link(link_data: LinkCreate, request: Request, user: dict = Depends(verify_api_key)):
    return await create_link(link_data, request, user)

@api_router.get("/v1/links")
async def api_get_links(user: dict = Depends(verify_api_key), skip: int = 0, limit: int = 50):
    return await get_links(user, skip, limit)

@api_router.get("/v1/links/{link_id}")
async def api_get_link(link_id: str, user: dict = Depends(verify_api_key)):
    return await get_link(link_id, user)

@api_router.get("/v1/links/{link_id}/analytics")
async def api_get_link_analytics(link_id: str, user: dict = Depends(verify_api_key)):
    return await get_link_analytics(link_id, user)

# ==================== SUBSCRIPTION ENDPOINTS ====================

@api_router.get("/plans")
async def get_plans():
    return {"plans": SUBSCRIPTION_PLANS}

@api_router.post("/checkout/subscribe")
async def create_subscription_checkout(checkout_data: SubscriptionCheckout, request: Request, user: dict = Depends(get_current_user)):
    from emergentintegrations.payments.stripe.checkout import StripeCheckout, CheckoutSessionRequest
    
    plan = SUBSCRIPTION_PLANS.get(checkout_data.plan_id)
    if not plan or plan["price"] == 0:
        raise HTTPException(status_code=400, detail="Invalid plan selected")
    
    host_url = str(request.base_url).rstrip('/')
    webhook_url = f"{host_url}/api/webhook/stripe"
    
    stripe_checkout = StripeCheckout(api_key=STRIPE_API_KEY, webhook_url=webhook_url)
    
    success_url = f"{checkout_data.origin_url}/settings?session_id={{CHECKOUT_SESSION_ID}}&status=success"
    cancel_url = f"{checkout_data.origin_url}/pricing?status=cancelled"
    
    payment_methods = ["card"]
    if checkout_data.payment_method == "crypto":
        payment_methods = ["card", "crypto"]
    
    checkout_request = CheckoutSessionRequest(
        amount=float(plan["price"]),
        currency="usd",
        success_url=success_url,
        cancel_url=cancel_url,
        metadata={
            "user_id": user["id"],
            "plan_id": checkout_data.plan_id,
            "type": "subscription"
        },
        payment_methods=payment_methods
    )
    
    session = await stripe_checkout.create_checkout_session(checkout_request)
    
    # Create payment transaction record
    transaction = {
        "id": str(uuid.uuid4()),
        "session_id": session.session_id,
        "user_id": user["id"],
        "plan_id": checkout_data.plan_id,
        "amount": float(plan["price"]),
        "currency": "usd",
        "payment_status": "pending",
        "type": "subscription",
        "created_at": datetime.now(timezone.utc).isoformat(),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.payment_transactions.insert_one(transaction)
    
    return {"url": session.url, "session_id": session.session_id}

@api_router.get("/checkout/status/{session_id}")
async def get_checkout_status(session_id: str, user: dict = Depends(get_current_user)):
    from emergentintegrations.payments.stripe.checkout import StripeCheckout
    
    host_url = "https://boltlytics.preview.emergentagent.com"
    webhook_url = f"{host_url}/api/webhook/stripe"
    
    stripe_checkout = StripeCheckout(api_key=STRIPE_API_KEY, webhook_url=webhook_url)
    
    status = await stripe_checkout.get_checkout_status(session_id)
    
    # Update transaction
    transaction = await db.payment_transactions.find_one({"session_id": session_id}, {"_id": 0})
    
    if transaction and transaction["payment_status"] != "paid" and status.payment_status == "paid":
        # Update transaction status
        await db.payment_transactions.update_one(
            {"session_id": session_id},
            {"$set": {
                "payment_status": "paid",
                "updated_at": datetime.now(timezone.utc).isoformat()
            }}
        )
        
        # Upgrade user's subscription
        plan_id = transaction.get("plan_id")
        if plan_id:
            await db.users.update_one(
                {"id": user["id"]},
                {"$set": {
                    "subscription_plan": plan_id,
                    "updated_at": datetime.now(timezone.utc).isoformat()
                }}
            )
    
    return {
        "status": status.status,
        "payment_status": status.payment_status,
        "amount_total": status.amount_total,
        "currency": status.currency
    }

@api_router.post("/webhook/stripe")
async def stripe_webhook(request: Request):
    from emergentintegrations.payments.stripe.checkout import StripeCheckout
    
    body = await request.body()
    signature = request.headers.get("Stripe-Signature")
    
    host_url = str(request.base_url).rstrip('/')
    webhook_url = f"{host_url}/api/webhook/stripe"
    
    stripe_checkout = StripeCheckout(api_key=STRIPE_API_KEY, webhook_url=webhook_url)
    
    try:
        event = await stripe_checkout.handle_webhook(body, signature)
        
        if event.payment_status == "paid":
            transaction = await db.payment_transactions.find_one(
                {"session_id": event.session_id},
                {"_id": 0}
            )
            
            if transaction and transaction["payment_status"] != "paid":
                await db.payment_transactions.update_one(
                    {"session_id": event.session_id},
                    {"$set": {
                        "payment_status": "paid",
                        "updated_at": datetime.now(timezone.utc).isoformat()
                    }}
                )
                
                # Upgrade subscription
                if transaction.get("type") == "subscription":
                    await db.users.update_one(
                        {"id": transaction["user_id"]},
                        {"$set": {
                            "subscription_plan": transaction["plan_id"],
                            "updated_at": datetime.now(timezone.utc).isoformat()
                        }}
                    )
        
        return {"status": "success"}
    except Exception as e:
        logging.error(f"Webhook error: {e}")
        return {"status": "error", "message": str(e)}

# ==================== ADMIN ENDPOINTS ====================

@api_router.get("/admin/users")
async def admin_get_users(admin: dict = Depends(get_admin_user), skip: int = 0, limit: int = 50):
    users = await db.users.find({}, {"_id": 0, "password": 0}).skip(skip).limit(limit).to_list(limit)
    total = await db.users.count_documents({})
    return {"users": users, "total": total}

@api_router.get("/admin/links")
async def admin_get_links(admin: dict = Depends(get_admin_user), skip: int = 0, limit: int = 50):
    links = await db.links.find({}, {"_id": 0, "password": 0}).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    total = await db.links.count_documents({})
    return {"links": links, "total": total}

@api_router.get("/admin/stats")
async def admin_get_stats(admin: dict = Depends(get_admin_user)):
    total_users = await db.users.count_documents({})
    total_links = await db.links.count_documents({})
    total_clicks = await db.clicks.count_documents({})
    
    # Revenue from paid transactions
    revenue_pipeline = [
        {"$match": {"payment_status": "paid"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    revenue_result = await db.payment_transactions.aggregate(revenue_pipeline).to_list(1)
    total_revenue = revenue_result[0]["total"] if revenue_result else 0
    
    # Users by plan
    plans_pipeline = [
        {"$group": {"_id": "$subscription_plan", "count": {"$sum": 1}}}
    ]
    users_by_plan = await db.users.aggregate(plans_pipeline).to_list(None)
    
    return {
        "total_users": total_users,
        "total_links": total_links,
        "total_clicks": total_clicks,
        "total_revenue": total_revenue,
        "users_by_plan": {p["_id"]: p["count"] for p in users_by_plan}
    }

@api_router.put("/admin/users/{user_id}/plan")
async def admin_update_user_plan(user_id: str, plan_id: str, admin: dict = Depends(get_admin_user)):
    if plan_id not in SUBSCRIPTION_PLANS:
        raise HTTPException(status_code=400, detail="Invalid plan")
    
    result = await db.users.update_one(
        {"id": user_id},
        {"$set": {"subscription_plan": plan_id, "updated_at": datetime.now(timezone.utc).isoformat()}}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="User not found")
    
    return {"message": "User plan updated successfully"}

@api_router.delete("/admin/users/{user_id}")
async def admin_delete_user(user_id: str, admin: dict = Depends(get_admin_user)):
    if user_id == admin["id"]:
        raise HTTPException(status_code=400, detail="Cannot delete yourself")
    
    result = await db.users.delete_one({"id": user_id})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Delete user's links and clicks
    user_links = await db.links.find({"user_id": user_id}, {"id": 1}).to_list(None)
    link_ids = [l["id"] for l in user_links]
    await db.links.delete_many({"user_id": user_id})
    await db.clicks.delete_many({"link_id": {"$in": link_ids}})
    await db.api_keys.delete_many({"user_id": user_id})
    
    return {"message": "User deleted successfully"}

# ==================== REDIRECT ENDPOINT ====================

@app.get("/s/{short_code}")
async def redirect_short_link(short_code: str, request: Request, password: Optional[str] = None):
    link = await db.links.find_one({"short_code": short_code}, {"_id": 0})
    
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    
    # Check expiration
    if link.get("expires_at"):
        expires = datetime.fromisoformat(link["expires_at"].replace("Z", "+00:00"))
        if datetime.now(timezone.utc) > expires:
            raise HTTPException(status_code=410, detail="Link has expired")
    
    # Check password
    if link.get("password"):
        if not password:
            # Return password required response
            return JSONResponse(
                status_code=401,
                content={"detail": "Password required", "requires_password": True}
            )
        if not verify_password(password, link["password"]):
            raise HTTPException(status_code=401, detail="Invalid password")
    
    # Parse request info
    user_agent = request.headers.get("User-Agent", "")
    request_info = parse_request_info(request, user_agent)
    
    # Get referrer
    referrer = request.headers.get("Referer", "Direct")
    
    # Record click
    click = {
        "id": str(uuid.uuid4()),
        "link_id": link["id"],
        "ip": request_info["ip"],
        "country": "Unknown",  # Would need GeoIP service
        "city": "Unknown",
        "browser": request_info["browser"],
        "browser_version": request_info["browser_version"],
        "os": request_info["os"],
        "device": request_info["device"],
        "referrer": referrer,
        "is_bot": request_info["is_bot"],
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.clicks.insert_one(click)
    
    # Increment click count
    await db.links.update_one(
        {"short_code": short_code},
        {"$inc": {"clicks": 1}}
    )
    
    return RedirectResponse(url=link["target_url"], status_code=302)

@app.post("/s/{short_code}/verify")
async def verify_link_password(short_code: str, data: PasswordVerify, request: Request):
    link = await db.links.find_one({"short_code": short_code}, {"_id": 0})
    
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    
    if not link.get("password"):
        return {"valid": True, "target_url": link["target_url"]}
    
    if verify_password(data.password, link["password"]):
        # Record click
        user_agent = request.headers.get("User-Agent", "")
        request_info = parse_request_info(request, user_agent)
        referrer = request.headers.get("Referer", "Direct")
        
        click = {
            "id": str(uuid.uuid4()),
            "link_id": link["id"],
            "ip": request_info["ip"],
            "country": "Unknown",
            "city": "Unknown",
            "browser": request_info["browser"],
            "browser_version": request_info["browser_version"],
            "os": request_info["os"],
            "device": request_info["device"],
            "referrer": referrer,
            "is_bot": request_info["is_bot"],
            "created_at": datetime.now(timezone.utc).isoformat()
        }
        
        await db.clicks.insert_one(click)
        await db.links.update_one({"short_code": short_code}, {"$inc": {"clicks": 1}})
        
        return {"valid": True, "target_url": link["target_url"]}
    
    raise HTTPException(status_code=401, detail="Invalid password")

# Get link info for interstitial page
@api_router.get("/public/link/{short_code}")
async def get_public_link_info(short_code: str):
    link = await db.links.find_one({"short_code": short_code}, {"_id": 0, "password": 0, "user_id": 0})
    
    if not link:
        raise HTTPException(status_code=404, detail="Link not found")
    
    # Check expiration
    if link.get("expires_at"):
        expires = datetime.fromisoformat(link["expires_at"].replace("Z", "+00:00"))
        if datetime.now(timezone.utc) > expires:
            raise HTTPException(status_code=410, detail="Link has expired")
    
    return {
        "short_code": link["short_code"],
        "title": link.get("title", ""),
        "target_url": link["target_url"],
        "requires_password": bool(await db.links.find_one({"short_code": short_code, "password": {"$ne": None}}))
    }

# Create admin user on startup
@app.on_event("startup")
async def create_admin_user():
    admin = await db.users.find_one({"email": "admin@boltlytics.com"})
    if not admin:
        admin_user = {
            "id": str(uuid.uuid4()),
            "email": "admin@boltlytics.com",
            "password": hash_password("admin123"),
            "name": "Admin",
            "role": "admin",
            "subscription_plan": "enterprise",
            "created_at": datetime.now(timezone.utc).isoformat(),
            "updated_at": datetime.now(timezone.utc).isoformat()
        }
        await db.users.insert_one(admin_user)
        logging.info("Admin user created: admin@boltlytics.com / admin123")

# Include the router
app.include_router(api_router)

app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@app.on_event("shutdown")
async def shutdown_db_client():
    client.close()
