import requests
import sys
import json
from datetime import datetime

class ShortlyticsAPITester:
    def __init__(self, base_url="https://shortlytics.preview.emergentagent.com"):
        self.base_url = base_url
        self.token = None
        self.admin_token = None
        self.tests_run = 0
        self.tests_passed = 0
        self.created_links = []

    def run_test(self, name, method, endpoint, expected_status, data=None, headers=None, use_admin=False):
        """Run a single API test"""
        url = f"{self.base_url}/api/{endpoint}"
        test_headers = {'Content-Type': 'application/json'}
        
        # Add auth token if available
        token = self.admin_token if use_admin and self.admin_token else self.token
        if token:
            test_headers['Authorization'] = f'Bearer {token}'
        
        if headers:
            test_headers.update(headers)

        self.tests_run += 1
        print(f"\n🔍 Testing {name}...")
        print(f"   URL: {url}")
        
        try:
            if method == 'GET':
                response = requests.get(url, headers=test_headers)
            elif method == 'POST':
                response = requests.post(url, json=data, headers=test_headers)
            elif method == 'PUT':
                response = requests.put(url, json=data, headers=test_headers)
            elif method == 'DELETE':
                response = requests.delete(url, headers=test_headers)

            success = response.status_code == expected_status
            if success:
                self.tests_passed += 1
                print(f"✅ Passed - Status: {response.status_code}")
                try:
                    return True, response.json()
                except:
                    return True, {}
            else:
                print(f"❌ Failed - Expected {expected_status}, got {response.status_code}")
                try:
                    error_detail = response.json()
                    print(f"   Error: {error_detail}")
                except:
                    print(f"   Response: {response.text}")
                return False, {}

        except Exception as e:
            print(f"❌ Failed - Error: {str(e)}")
            return False, {}

    def test_plans_endpoint(self):
        """Test subscription plans endpoint"""
        success, response = self.run_test(
            "Get Subscription Plans",
            "GET",
            "plans",
            200
        )
        if success and 'plans' in response:
            plans = response['plans']
            expected_plans = ['free', 'basic', 'pro', 'ultimate', 'enterprise']
            for plan in expected_plans:
                if plan not in plans:
                    print(f"❌ Missing plan: {plan}")
                    return False
            print(f"   Found {len(plans)} plans: {list(plans.keys())}")
        return success

    def test_admin_login(self):
        """Test admin login"""
        success, response = self.run_test(
            "Admin Login",
            "POST",
            "auth/login",
            200,
            data={"email": "admin@shortlytics.com", "password": "admin123"}
        )
        if success and 'token' in response:
            self.admin_token = response['token']
            print(f"   Admin role: {response.get('user', {}).get('role')}")
            return True
        return False

    def test_user_registration(self):
        """Test user registration"""
        timestamp = datetime.now().strftime('%H%M%S')
        test_user = {
            "email": f"test_user_{timestamp}@example.com",
            "password": "TestPass123!",
            "name": f"Test User {timestamp}"
        }
        
        success, response = self.run_test(
            "User Registration",
            "POST",
            "auth/register",
            200,
            data=test_user
        )
        if success and 'token' in response:
            self.token = response['token']
            print(f"   User ID: {response.get('user', {}).get('id')}")
            return True
        return False

    def test_user_login(self):
        """Test user login with registered credentials"""
        timestamp = datetime.now().strftime('%H%M%S')
        login_data = {
            "email": f"test_user_{timestamp}@example.com",
            "password": "TestPass123!"
        }
        
        success, response = self.run_test(
            "User Login",
            "POST",
            "auth/login",
            200,
            data=login_data
        )
        return success

    def test_get_user_profile(self):
        """Test get current user profile"""
        success, response = self.run_test(
            "Get User Profile",
            "GET",
            "auth/me",
            200
        )
        if success:
            print(f"   Plan: {response.get('subscription_plan')}")
            print(f"   Links this month: {response.get('links_created_this_month')}")
        return success

    def test_create_link(self):
        """Test creating a short link"""
        link_data = {
            "target_url": "https://example.com/test-page",
            "title": "Test Link",
            "custom_alias": f"test-{datetime.now().strftime('%H%M%S')}"
        }
        
        success, response = self.run_test(
            "Create Link",
            "POST",
            "links",
            200,
            data=link_data
        )
        if success and 'id' in response:
            self.created_links.append(response)
            print(f"   Short URL: {response.get('short_url')}")
            print(f"   Link ID: {response.get('id')}")
        return success

    def test_get_links(self):
        """Test getting user's links"""
        success, response = self.run_test(
            "Get User Links",
            "GET",
            "links",
            200
        )
        if success:
            print(f"   Total links: {response.get('total', 0)}")
        return success

    def test_link_analytics(self):
        """Test getting link analytics"""
        if not self.created_links:
            print("⚠️  No links created, skipping analytics test")
            return True
            
        link_id = self.created_links[0]['id']
        success, response = self.run_test(
            "Get Link Analytics",
            "GET",
            f"links/{link_id}/analytics",
            200
        )
        if success:
            print(f"   Total clicks: {response.get('total_clicks', 0)}")
        return success

    def test_analytics_overview(self):
        """Test getting analytics overview"""
        success, response = self.run_test(
            "Get Analytics Overview",
            "GET",
            "analytics/overview",
            200
        )
        if success:
            print(f"   Total links: {response.get('total_links', 0)}")
            print(f"   Total clicks: {response.get('total_clicks', 0)}")
        return success

    def test_admin_stats(self):
        """Test admin statistics endpoint"""
        success, response = self.run_test(
            "Get Admin Stats",
            "GET",
            "admin/stats",
            200,
            use_admin=True
        )
        if success:
            print(f"   Total users: {response.get('total_users', 0)}")
            print(f"   Total revenue: ${response.get('total_revenue', 0)}")
        return success

    def test_admin_users(self):
        """Test admin users endpoint"""
        success, response = self.run_test(
            "Get Admin Users",
            "GET",
            "admin/users",
            200,
            use_admin=True
        )
        if success:
            print(f"   Users found: {len(response.get('users', []))}")
        return success

    def test_api_key_creation_without_pro(self):
        """Test API key creation should fail without Pro plan"""
        success, response = self.run_test(
            "Create API Key (Should Fail - No Pro Plan)",
            "POST",
            "api-keys",
            403,  # Should fail with 403
            data={"name": "Test API Key"}
        )
        # For this test, success means getting the expected 403 error
        return success

    def test_password_protected_link(self):
        """Test creating a password protected link"""
        link_data = {
            "target_url": "https://example.com/secret-page",
            "title": "Secret Link",
            "password": "secret123"
        }
        
        success, response = self.run_test(
            "Create Password Protected Link",
            "POST",
            "links",
            200,
            data=link_data
        )
        if success:
            print(f"   Password protected: {response.get('is_password_protected')}")
        return success

def main():
    print("🚀 Starting Shortlytics API Tests")
    print("=" * 50)
    
    tester = ShortlyticsAPITester()
    
    # Test sequence
    tests = [
        ("Subscription Plans", tester.test_plans_endpoint),
        ("Admin Login", tester.test_admin_login),
        ("User Registration", tester.test_user_registration),
        ("User Profile", tester.test_get_user_profile),
        ("Create Link", tester.test_create_link),
        ("Get Links", tester.test_get_links),
        ("Link Analytics", tester.test_link_analytics),
        ("Analytics Overview", tester.test_analytics_overview),
        ("Password Protected Link", tester.test_password_protected_link),
        ("API Key Creation (No Pro)", tester.test_api_key_creation_without_pro),
        ("Admin Stats", tester.test_admin_stats),
        ("Admin Users", tester.test_admin_users),
    ]
    
    failed_tests = []
    
    for test_name, test_func in tests:
        try:
            if not test_func():
                failed_tests.append(test_name)
        except Exception as e:
            print(f"❌ {test_name} - Exception: {str(e)}")
            failed_tests.append(test_name)
    
    # Print results
    print("\n" + "=" * 50)
    print(f"📊 Test Results: {tester.tests_passed}/{tester.tests_run} passed")
    
    if failed_tests:
        print(f"\n❌ Failed Tests ({len(failed_tests)}):")
        for test in failed_tests:
            print(f"   - {test}")
    else:
        print("\n✅ All tests passed!")
    
    success_rate = (tester.tests_passed / tester.tests_run * 100) if tester.tests_run > 0 else 0
    print(f"\n📈 Success Rate: {success_rate:.1f}%")
    
    return 0 if len(failed_tests) == 0 else 1

if __name__ == "__main__":
    sys.exit(main())