import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import DashboardLayout from '../layouts/DashboardLayout';
import { api } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { toast } from 'sonner';
import {
  LineChart,
  Line,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  Legend,
} from 'recharts';
import {
  ArrowLeft,
  MousePointer,
  Globe,
  Monitor,
  Smartphone,
  Link2,
  ExternalLink,
  Copy,
} from 'lucide-react';

const COLORS = ['hsl(217, 91%, 60%)', 'hsl(160, 60%, 45%)', 'hsl(45, 93%, 58%)', 'hsl(0, 90%, 65%)', 'hsl(270, 70%, 70%)'];

const AnalyticsPage = () => {
  const { linkId } = useParams();
  const navigate = useNavigate();
  const [link, setLink] = useState(null);
  const [analytics, setAnalytics] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchData();
  }, [linkId]);

  const fetchData = async () => {
    try {
      const [linkRes, analyticsRes] = await Promise.all([
        api.get(`/links/${linkId}`),
        api.get(`/links/${linkId}/analytics`),
      ]);
      setLink(linkRes.data);
      setAnalytics(analyticsRes.data);
    } catch (error) {
      toast.error('Failed to load analytics');
      navigate('/links');
    } finally {
      setLoading(false);
    }
  };

  const copyToClipboard = (text) => {
    navigator.clipboard.writeText(text);
    toast.success('Copied to clipboard!');
  };

  const getShortUrl = () => {
    const baseUrl = process.env.REACT_APP_BACKEND_URL;
    return `${baseUrl}/s/${link?.short_code}`;
  };

  if (loading) {
    return (
      <DashboardLayout>
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
        </div>
      </DashboardLayout>
    );
  }

  const stats = [
    { title: 'Total Clicks', value: analytics?.total_clicks || 0, icon: MousePointer },
    { title: 'Today', value: analytics?.clicks_today || 0, icon: MousePointer },
    { title: 'This Week', value: analytics?.clicks_this_week || 0, icon: MousePointer },
    { title: 'This Month', value: analytics?.clicks_this_month || 0, icon: MousePointer },
  ];

  return (
    <DashboardLayout>
      <div className="space-y-6 fade-in">
        {/* Header */}
        <div className="flex flex-col md:flex-row md:items-center gap-4">
          <Button variant="ghost" onClick={() => navigate('/links')} data-testid="back-btn">
            <ArrowLeft className="mr-2 h-4 w-4" />
            Back to Links
          </Button>
          <div className="flex-1">
            <h1 className="text-2xl font-bold">{link?.title || link?.short_code}</h1>
            <div className="flex items-center gap-2 mt-1">
              <code className="text-sm bg-muted px-2 py-1 rounded font-mono text-primary">
                {getShortUrl()}
              </code>
              <Button
                variant="ghost"
                size="icon"
                onClick={() => copyToClipboard(getShortUrl())}
                data-testid="copy-link"
              >
                <Copy className="h-4 w-4" />
              </Button>
              <Button
                variant="ghost"
                size="icon"
                onClick={() => window.open(link?.target_url, '_blank')}
                data-testid="visit-original"
              >
                <ExternalLink className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          {stats.map((stat, index) => (
            <Card key={index} className="border-2">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <stat.icon className="h-6 w-6 text-primary" />
                  <span className="text-3xl font-bold">{stat.value.toLocaleString()}</span>
                </div>
                <p className="text-sm text-muted-foreground mt-2">{stat.title}</p>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Charts Grid */}
        <div className="grid md:grid-cols-2 gap-6">
          {/* Clicks Over Time */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle>Clicks Over Time</CardTitle>
              <CardDescription>Last 30 days</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-64">
                {analytics?.clicks_by_day?.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <LineChart data={analytics.clicks_by_day}>
                      <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                      <XAxis
                        dataKey="date"
                        tickFormatter={(value) => value.slice(5)}
                        className="text-xs"
                      />
                      <YAxis className="text-xs" />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px',
                        }}
                      />
                      <Line
                        type="monotone"
                        dataKey="count"
                        stroke="hsl(var(--primary))"
                        strokeWidth={2}
                        dot={false}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="h-full flex items-center justify-center text-muted-foreground">
                    No click data yet
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Top Countries */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Globe className="h-5 w-5" />
                Top Countries
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="h-64">
                {analytics?.top_countries?.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <BarChart data={analytics.top_countries} layout="vertical">
                      <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                      <XAxis type="number" className="text-xs" />
                      <YAxis
                        dataKey="country"
                        type="category"
                        width={80}
                        className="text-xs"
                      />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px',
                        }}
                      />
                      <Bar dataKey="count" fill="hsl(var(--primary))" radius={[0, 4, 4, 0]} />
                    </BarChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="h-full flex items-center justify-center text-muted-foreground">
                    No location data yet
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Devices */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Monitor className="h-5 w-5" />
                Devices
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="h-64">
                {analytics?.top_devices?.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <PieChart>
                      <Pie
                        data={analytics.top_devices}
                        dataKey="count"
                        nameKey="device"
                        cx="50%"
                        cy="50%"
                        outerRadius={80}
                        label={({ device, percent }) =>
                          `${device} ${(percent * 100).toFixed(0)}%`
                        }
                      >
                        {analytics.top_devices.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Pie>
                      <Tooltip
                        contentStyle={{
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px',
                        }}
                      />
                    </PieChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="h-full flex items-center justify-center text-muted-foreground">
                    No device data yet
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Browsers */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle>Browsers</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="h-64">
                {analytics?.top_browsers?.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <PieChart>
                      <Pie
                        data={analytics.top_browsers}
                        dataKey="count"
                        nameKey="browser"
                        cx="50%"
                        cy="50%"
                        outerRadius={80}
                        label={({ browser, percent }) =>
                          `${browser} ${(percent * 100).toFixed(0)}%`
                        }
                      >
                        {analytics.top_browsers.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Pie>
                      <Tooltip
                        contentStyle={{
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px',
                        }}
                      />
                    </PieChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="h-full flex items-center justify-center text-muted-foreground">
                    No browser data yet
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Referrers Table */}
        <Card className="border-2">
          <CardHeader>
            <CardTitle>Top Referrers</CardTitle>
            <CardDescription>Where your clicks are coming from</CardDescription>
          </CardHeader>
          <CardContent>
            {analytics?.top_referrers?.length > 0 ? (
              <div className="space-y-4">
                {analytics.top_referrers.map((ref, index) => (
                  <div
                    key={index}
                    className="flex items-center justify-between p-3 bg-muted rounded-lg"
                  >
                    <div className="flex items-center gap-3">
                      <span className="text-lg font-bold text-muted-foreground">
                        #{index + 1}
                      </span>
                      <span className="font-medium">{ref.referrer}</span>
                    </div>
                    <span className="text-lg font-bold">{ref.count}</span>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-8 text-muted-foreground">
                No referrer data yet
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
};

export default AnalyticsPage;
