import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import DashboardLayout from '../layouts/DashboardLayout';
import { useAuth, api } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Input } from '../components/ui/input';
import { toast } from 'sonner';
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
} from 'recharts';
import {
  Link2,
  MousePointer,
  TrendingUp,
  Plus,
  ArrowUpRight,
  Copy,
  ExternalLink,
} from 'lucide-react';

const Dashboard = () => {
  const { user } = useAuth();
  const navigate = useNavigate();
  const [overview, setOverview] = useState(null);
  const [loading, setLoading] = useState(true);
  const [quickUrl, setQuickUrl] = useState('');
  const [creating, setCreating] = useState(false);

  useEffect(() => {
    fetchOverview();
  }, []);

  const fetchOverview = async () => {
    try {
      const res = await api.get('/analytics/overview');
      setOverview(res.data);
    } catch (error) {
      toast.error('Failed to load dashboard data');
    } finally {
      setLoading(false);
    }
  };

  const handleQuickCreate = async (e) => {
    e.preventDefault();
    if (!quickUrl) return;

    setCreating(true);
    try {
      const res = await api.post('/links', { target_url: quickUrl });
      toast.success('Link created successfully!');
      setQuickUrl('');
      fetchOverview();
      navigate('/links');
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to create link');
    } finally {
      setCreating(false);
    }
  };

  const copyToClipboard = (text) => {
    navigator.clipboard.writeText(text);
    toast.success('Copied to clipboard!');
  };

  const stats = overview
    ? [
        {
          title: 'Total Links',
          value: overview.total_links,
          icon: Link2,
          color: 'text-primary',
        },
        {
          title: 'Total Clicks',
          value: overview.total_clicks,
          icon: MousePointer,
          color: 'text-green-500',
        },
        {
          title: 'Clicks Today',
          value: overview.clicks_today,
          icon: TrendingUp,
          color: 'text-yellow-500',
        },
        {
          title: 'This Month',
          value: overview.clicks_this_month,
          icon: ArrowUpRight,
          color: 'text-purple-500',
        },
      ]
    : [];

  if (loading) {
    return (
      <DashboardLayout title="Dashboard">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout title="Dashboard">
      <div className="space-y-6 fade-in">
        {/* Welcome Section */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <h2 className="text-xl font-semibold">Welcome back, {user?.name}!</h2>
            <p className="text-muted-foreground">
              {user?.links_created_this_month || 0} / {user?.links_limit === -1 ? '∞' : user?.links_limit} links used this month
            </p>
          </div>
          <Button onClick={() => navigate('/links')} data-testid="create-link-btn">
            <Plus className="mr-2 h-4 w-4" />
            Create Link
          </Button>
        </div>

        {/* Quick Create */}
        <Card className="border-2">
          <CardHeader className="pb-4">
            <CardTitle className="text-lg">Quick Shorten</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleQuickCreate} className="flex gap-3">
              <Input
                placeholder="Paste your long URL here..."
                value={quickUrl}
                onChange={(e) => setQuickUrl(e.target.value)}
                className="flex-1"
                data-testid="quick-url-input"
              />
              <Button type="submit" disabled={creating} data-testid="quick-create-btn">
                {creating ? 'Creating...' : 'Shorten'}
              </Button>
            </form>
          </CardContent>
        </Card>

        {/* Stats Grid */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          {stats.map((stat, index) => (
            <Card key={index} className="border-2 hover-lift">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <stat.icon className={`h-8 w-8 ${stat.color}`} />
                  <span className="text-3xl font-bold">{stat.value.toLocaleString()}</span>
                </div>
                <p className="text-sm text-muted-foreground mt-2">{stat.title}</p>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Charts Row */}
        <div className="grid md:grid-cols-2 gap-6">
          {/* Clicks Over Time */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle>Clicks Over Time</CardTitle>
              <CardDescription>Last 30 days performance</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-64">
                {overview?.clicks_by_day?.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <LineChart data={overview.clicks_by_day}>
                      <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                      <XAxis
                        dataKey="date"
                        tickFormatter={(value) => value.slice(5)}
                        className="text-xs"
                      />
                      <YAxis className="text-xs" />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px',
                        }}
                      />
                      <Line
                        type="monotone"
                        dataKey="count"
                        stroke="hsl(var(--primary))"
                        strokeWidth={2}
                        dot={false}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="h-full flex items-center justify-center text-muted-foreground">
                    No click data yet
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Top Links */}
          <Card className="border-2">
            <CardHeader>
              <CardTitle>Top Performing Links</CardTitle>
              <CardDescription>Your most clicked links</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {overview?.top_links?.length > 0 ? (
                  overview.top_links.map((link, index) => (
                    <div
                      key={link.id}
                      className="flex items-center justify-between p-3 bg-muted rounded-lg"
                    >
                      <div className="flex items-center gap-3 min-w-0">
                        <span className="text-lg font-bold text-muted-foreground">
                          #{index + 1}
                        </span>
                        <div className="min-w-0">
                          <p className="font-medium truncate">{link.title || link.short_code}</p>
                          <p className="text-sm text-muted-foreground font-mono">
                            /{link.short_code}
                          </p>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="text-lg font-bold">{link.clicks}</span>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => navigate(`/analytics/${link.id}`)}
                          data-testid={`view-analytics-${link.short_code}`}
                        >
                          <ArrowUpRight className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                  ))
                ) : (
                  <div className="text-center py-8 text-muted-foreground">
                    <Link2 className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>No links yet</p>
                    <Button
                      variant="link"
                      onClick={() => navigate('/links')}
                      className="mt-2"
                    >
                      Create your first link
                    </Button>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
};

export default Dashboard;
