import React, { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { useTheme } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { 
  Link2, 
  BarChart3, 
  Lock, 
  Globe, 
  Zap, 
  Moon, 
  Sun, 
  Check,
  ArrowRight,
  QrCode,
  Key,
  Target,
  Shield,
  Clock
} from 'lucide-react';

const PLANS = {
  free: { name: 'Free', price: 0, links: 3, features: ['Basic analytics', 'Standard support'] },
  basic: { name: 'Basic', price: 9.99, links: 30, features: ['Advanced analytics', 'Priority support', 'QR codes'] },
  pro: { name: 'Pro', price: 29.99, links: 300, features: ['Full analytics', 'API access', 'Password protection', 'Custom aliases'] },
  ultimate: { name: 'Ultimate', price: 79.99, links: 1000, features: ['Everything in Pro', 'Custom domains', 'White-label', 'Dedicated support'] },
};

const LandingPage = () => {
  const { theme, toggleTheme } = useTheme();
  const navigate = useNavigate();

  const features = [
    {
      icon: Link2,
      title: 'Powerful Link Shortening',
      description: 'Create short, memorable links with custom aliases that work anywhere.'
    },
    {
      icon: BarChart3,
      title: 'Advanced Analytics',
      description: 'Track clicks, locations, devices, and referrers with detailed insights.'
    },
    {
      icon: Lock,
      title: 'Password Protection',
      description: 'Secure your links with passwords for private content sharing.'
    },
    {
      icon: Globe,
      title: 'Location Targeting',
      description: 'Restrict access based on geographic location for targeted campaigns.'
    },
    {
      icon: QrCode,
      title: 'QR Code Generation',
      description: 'Auto-generate QR codes for every link, perfect for print and offline.'
    },
    {
      icon: Key,
      title: 'API Access',
      description: 'Integrate link shortening into your apps with our powerful API.'
    },
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="sticky top-0 z-50 glass border-b">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="flex h-16 items-center justify-between">
            <Link to="/" className="flex items-center gap-2" data-testid="logo">
              <Link2 className="h-6 w-6 text-primary" />
              <span className="font-bold text-xl tracking-tight">Boltlytics</span>
            </Link>

            <nav className="hidden md:flex items-center gap-8">
              <a href="#features" className="text-muted-foreground hover:text-foreground transition-colors">
                Features
              </a>
              <a href="#pricing" className="text-muted-foreground hover:text-foreground transition-colors">
                Pricing
              </a>
              <Link to="/login" className="text-muted-foreground hover:text-foreground transition-colors">
                Login
              </Link>
            </nav>

            <div className="flex items-center gap-3">
              <Button
                variant="ghost"
                size="icon"
                onClick={toggleTheme}
                data-testid="theme-toggle-landing"
              >
                {theme === 'dark' ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}
              </Button>
              <Button onClick={() => navigate('/register')} data-testid="get-started-btn">
                Get Started
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="py-20 md:py-32 px-4 md:px-8">
        <div className="max-w-7xl mx-auto">
          <div className="grid md:grid-cols-12 gap-12 items-center">
            <div className="md:col-span-7 space-y-8">
              <h1 className="text-5xl md:text-7xl font-black tracking-tighter leading-none">
                Shorten Links.
                <br />
                <span className="text-primary">Track Everything.</span>
              </h1>
              <p className="text-lg md:text-xl text-muted-foreground max-w-xl leading-relaxed">
                The most powerful URL shortener with enterprise-grade analytics. 
                Perfect for marketers, developers, and businesses.
              </p>
              <div className="flex flex-col sm:flex-row gap-4">
                <Button size="lg" onClick={() => navigate('/register')} data-testid="hero-cta">
                  Start Free
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
                <Button size="lg" variant="outline" onClick={() => navigate('/pricing')} data-testid="view-pricing-btn">
                  View Pricing
                </Button>
              </div>
              <div className="flex items-center gap-6 text-sm text-muted-foreground">
                <div className="flex items-center gap-2">
                  <Check className="h-4 w-4 text-primary" />
                  <span>Free forever plan</span>
                </div>
                <div className="flex items-center gap-2">
                  <Check className="h-4 w-4 text-primary" />
                  <span>No credit card required</span>
                </div>
              </div>
            </div>
            <div className="md:col-span-5">
              <Card className="border-2 shadow-2xl">
                <CardHeader className="pb-4">
                  <div className="flex items-center gap-2 text-xs text-muted-foreground mb-2">
                    <div className="h-3 w-3 rounded-full bg-destructive" />
                    <div className="h-3 w-3 rounded-full bg-yellow-500" />
                    <div className="h-3 w-3 rounded-full bg-green-500" />
                  </div>
                  <div className="flex items-center gap-2 bg-muted rounded-md px-3 py-2">
                    <Link2 className="h-4 w-4 text-muted-foreground" />
                    <span className="text-sm text-primary font-mono">srtly.io/demo</span>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="bg-muted rounded-lg p-4">
                      <p className="text-3xl font-bold text-primary">12.4K</p>
                      <p className="text-xs text-muted-foreground">Total Clicks</p>
                    </div>
                    <div className="bg-muted rounded-lg p-4">
                      <p className="text-3xl font-bold text-green-500">+24%</p>
                      <p className="text-xs text-muted-foreground">This Week</p>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Top Country</span>
                      <span className="font-medium">United States</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Top Referrer</span>
                      <span className="font-medium">Twitter</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Top Device</span>
                      <span className="font-medium">Mobile</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section id="features" className="py-20 bg-muted/30">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold tracking-tight mb-4">
              Everything You Need
            </h2>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Powerful features to help you manage, track, and optimize your links.
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {features.map((feature, index) => (
              <Card key={index} className="hover-lift border-2 hover:border-primary/50 transition-colors">
                <CardHeader>
                  <div className="h-12 w-12 rounded-lg bg-primary/10 flex items-center justify-center mb-4">
                    <feature.icon className="h-6 w-6 text-primary" />
                  </div>
                  <CardTitle className="text-xl">{feature.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-muted-foreground">{feature.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Stats Section */}
      <section className="py-20">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
            {[
              { value: '10M+', label: 'Links Created' },
              { value: '500M+', label: 'Clicks Tracked' },
              { value: '50K+', label: 'Active Users' },
              { value: '99.9%', label: 'Uptime' },
            ].map((stat, index) => (
              <div key={index} className="text-center">
                <p className="text-4xl md:text-5xl font-black text-primary">{stat.value}</p>
                <p className="text-muted-foreground mt-2">{stat.label}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing" className="py-20 bg-muted/30">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold tracking-tight mb-4">
              Simple, Transparent Pricing
            </h2>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Choose the plan that fits your needs. Upgrade or downgrade anytime.
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {Object.entries(PLANS).map(([key, plan]) => (
              <Card 
                key={key} 
                className={`hover-lift ${key === 'pro' ? 'border-primary border-2 shadow-lg shadow-primary/20' : 'border-2'}`}
              >
                <CardHeader>
                  {key === 'pro' && (
                    <div className="text-xs font-medium text-primary mb-2">MOST POPULAR</div>
                  )}
                  <CardTitle className="text-2xl">{plan.name}</CardTitle>
                  <CardDescription>
                    <span className="text-4xl font-bold text-foreground">${plan.price}</span>
                    <span className="text-muted-foreground">/month</span>
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="text-sm font-medium">
                    {plan.links === -1 ? 'Unlimited' : plan.links} links/month
                  </div>
                  <ul className="space-y-2">
                    {plan.features.map((feature, i) => (
                      <li key={i} className="flex items-center gap-2 text-sm">
                        <Check className="h-4 w-4 text-primary" />
                        <span>{feature}</span>
                      </li>
                    ))}
                  </ul>
                  <Button 
                    className="w-full" 
                    variant={key === 'pro' ? 'default' : 'outline'}
                    onClick={() => navigate('/register')}
                    data-testid={`pricing-${key}-btn`}
                  >
                    {plan.price === 0 ? 'Get Started' : 'Subscribe'}
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          <div className="text-center mt-12">
            <Card className="inline-block border-2">
              <CardContent className="flex items-center gap-6 p-6">
                <Shield className="h-12 w-12 text-primary" />
                <div className="text-left">
                  <h3 className="text-xl font-bold">Enterprise</h3>
                  <p className="text-muted-foreground">Custom solutions for large organizations</p>
                </div>
                <Button variant="outline" onClick={() => navigate('/contact')}>
                  Contact Sales
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20">
        <div className="max-w-3xl mx-auto px-4 md:px-8 text-center">
          <h2 className="text-4xl md:text-5xl font-bold tracking-tight mb-6">
            Ready to Get Started?
          </h2>
          <p className="text-lg text-muted-foreground mb-8">
            Join thousands of marketers and developers using Boltlytics to manage their links.
          </p>
          <Button size="lg" onClick={() => navigate('/register')} data-testid="final-cta">
            Create Free Account
            <ArrowRight className="ml-2 h-4 w-4" />
          </Button>
        </div>
      </section>

      {/* Footer */}
      <footer className="border-t py-12">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="flex flex-col md:flex-row justify-between items-center gap-4">
            <div className="flex items-center gap-2">
              <Link2 className="h-5 w-5 text-primary" />
              <span className="font-bold">Boltlytics</span>
            </div>
            <p className="text-sm text-muted-foreground">
              © 2024 Boltlytics. All rights reserved.
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
};

export default LandingPage;
