import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import DashboardLayout from '../layouts/DashboardLayout';
import { api } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Input } from '../components/ui/input';
import { Label } from '../components/ui/label';
import { Switch } from '../components/ui/switch';
import { Calendar } from '../components/ui/calendar';
import { Popover, PopoverContent, PopoverTrigger } from '../components/ui/popover';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
} from '../components/ui/dialog';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '../components/ui/dropdown-menu';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '../components/ui/table';
import { toast } from 'sonner';
import { format } from 'date-fns';
import {
  Plus,
  Copy,
  ExternalLink,
  BarChart3,
  MoreVertical,
  Trash2,
  Edit,
  Lock,
  Calendar as CalendarIcon,
  QrCode,
  Link2,
  Search,
} from 'lucide-react';

const LinksPage = () => {
  const navigate = useNavigate();
  const [links, setLinks] = useState([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState('');
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [qrDialogOpen, setQrDialogOpen] = useState(false);
  const [selectedLink, setSelectedLink] = useState(null);
  
  // Create form state
  const [targetUrl, setTargetUrl] = useState('');
  const [customAlias, setCustomAlias] = useState('');
  const [title, setTitle] = useState('');
  const [password, setPassword] = useState('');
  const [usePassword, setUsePassword] = useState(false);
  const [expiresAt, setExpiresAt] = useState(null);
  const [useExpiry, setUseExpiry] = useState(false);
  const [creating, setCreating] = useState(false);

  useEffect(() => {
    fetchLinks();
  }, []);

  const fetchLinks = async () => {
    try {
      const res = await api.get('/links');
      setLinks(res.data.links);
    } catch (error) {
      toast.error('Failed to load links');
    } finally {
      setLoading(false);
    }
  };

  const handleCreate = async (e) => {
    e.preventDefault();
    if (!targetUrl) {
      toast.error('Please enter a URL');
      return;
    }

    setCreating(true);
    try {
      const payload = {
        target_url: targetUrl,
        title: title || undefined,
        custom_alias: customAlias || undefined,
        password: usePassword ? password : undefined,
        expires_at: useExpiry && expiresAt ? expiresAt.toISOString() : undefined,
      };

      await api.post('/links', payload);
      toast.success('Link created successfully!');
      setCreateDialogOpen(false);
      resetForm();
      fetchLinks();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to create link');
    } finally {
      setCreating(false);
    }
  };

  const handleDelete = async (linkId) => {
    if (!confirm('Are you sure you want to delete this link?')) return;

    try {
      await api.delete(`/links/${linkId}`);
      toast.success('Link deleted');
      fetchLinks();
    } catch (error) {
      toast.error('Failed to delete link');
    }
  };

  const resetForm = () => {
    setTargetUrl('');
    setCustomAlias('');
    setTitle('');
    setPassword('');
    setUsePassword(false);
    setExpiresAt(null);
    setUseExpiry(false);
  };

  const copyToClipboard = (text) => {
    navigator.clipboard.writeText(text);
    toast.success('Copied to clipboard!');
  };

  const getShortUrl = (shortCode) => {
    const baseUrl = process.env.REACT_APP_BACKEND_URL;
    return `${baseUrl}/s/${shortCode}`;
  };

  const filteredLinks = links.filter(
    (link) =>
      link.title?.toLowerCase().includes(search.toLowerCase()) ||
      link.short_code.toLowerCase().includes(search.toLowerCase()) ||
      link.target_url.toLowerCase().includes(search.toLowerCase())
  );

  if (loading) {
    return (
      <DashboardLayout title="Links">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout title="Links">
      <div className="space-y-6 fade-in">
        {/* Actions Bar */}
        <div className="flex flex-col md:flex-row gap-4 md:items-center md:justify-between">
          <div className="relative flex-1 max-w-sm">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search links..."
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              className="pl-9"
              data-testid="search-links"
            />
          </div>
          
          <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
            <DialogTrigger asChild>
              <Button data-testid="create-link-btn">
                <Plus className="mr-2 h-4 w-4" />
                Create Link
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle>Create New Link</DialogTitle>
                <DialogDescription>
                  Shorten a URL and track its performance
                </DialogDescription>
              </DialogHeader>
              
              <form onSubmit={handleCreate} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="url">Destination URL *</Label>
                  <Input
                    id="url"
                    placeholder="https://example.com/long-url"
                    value={targetUrl}
                    onChange={(e) => setTargetUrl(e.target.value)}
                    required
                    data-testid="create-url-input"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="title">Title (optional)</Label>
                  <Input
                    id="title"
                    placeholder="My awesome link"
                    value={title}
                    onChange={(e) => setTitle(e.target.value)}
                    data-testid="create-title-input"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="alias">Custom Alias (optional)</Label>
                  <Input
                    id="alias"
                    placeholder="my-custom-link"
                    value={customAlias}
                    onChange={(e) => setCustomAlias(e.target.value)}
                    data-testid="create-alias-input"
                  />
                </div>

                <div className="flex items-center justify-between">
                  <Label htmlFor="password-toggle" className="flex items-center gap-2">
                    <Lock className="h-4 w-4" />
                    Password Protection
                  </Label>
                  <Switch
                    id="password-toggle"
                    checked={usePassword}
                    onCheckedChange={setUsePassword}
                    data-testid="password-toggle"
                  />
                </div>

                {usePassword && (
                  <Input
                    type="password"
                    placeholder="Enter password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    data-testid="create-password-input"
                  />
                )}

                <div className="flex items-center justify-between">
                  <Label htmlFor="expiry-toggle" className="flex items-center gap-2">
                    <CalendarIcon className="h-4 w-4" />
                    Set Expiration
                  </Label>
                  <Switch
                    id="expiry-toggle"
                    checked={useExpiry}
                    onCheckedChange={setUseExpiry}
                    data-testid="expiry-toggle"
                  />
                </div>

                {useExpiry && (
                  <Popover>
                    <PopoverTrigger asChild>
                      <Button variant="outline" className="w-full justify-start">
                        <CalendarIcon className="mr-2 h-4 w-4" />
                        {expiresAt ? format(expiresAt, 'PPP') : 'Pick a date'}
                      </Button>
                    </PopoverTrigger>
                    <PopoverContent className="w-auto p-0" align="start">
                      <Calendar
                        mode="single"
                        selected={expiresAt}
                        onSelect={setExpiresAt}
                        disabled={(date) => date < new Date()}
                        initialFocus
                      />
                    </PopoverContent>
                  </Popover>
                )}

                <DialogFooter>
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setCreateDialogOpen(false)}
                  >
                    Cancel
                  </Button>
                  <Button type="submit" disabled={creating} data-testid="submit-create-link">
                    {creating ? 'Creating...' : 'Create Link'}
                  </Button>
                </DialogFooter>
              </form>
            </DialogContent>
          </Dialog>
        </div>

        {/* Links Table */}
        <Card className="border-2">
          <CardContent className="p-0">
            {filteredLinks.length > 0 ? (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Link</TableHead>
                    <TableHead className="hidden md:table-cell">Destination</TableHead>
                    <TableHead className="text-center">Clicks</TableHead>
                    <TableHead className="hidden md:table-cell">Created</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredLinks.map((link) => (
                    <TableRow key={link.id}>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {link.is_password_protected && (
                            <Lock className="h-4 w-4 text-muted-foreground" />
                          )}
                          <div>
                            <p className="font-medium">{link.title || link.short_code}</p>
                            <p className="text-sm text-primary font-mono">
                              /{link.short_code}
                            </p>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="hidden md:table-cell max-w-xs">
                        <p className="truncate text-sm text-muted-foreground">
                          {link.target_url}
                        </p>
                      </TableCell>
                      <TableCell className="text-center">
                        <span className="font-bold">{link.clicks}</span>
                      </TableCell>
                      <TableCell className="hidden md:table-cell text-sm text-muted-foreground">
                        {new Date(link.created_at).toLocaleDateString()}
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex items-center justify-end gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => copyToClipboard(getShortUrl(link.short_code))}
                            data-testid={`copy-${link.short_code}`}
                          >
                            <Copy className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => {
                              setSelectedLink(link);
                              setQrDialogOpen(true);
                            }}
                            data-testid={`qr-${link.short_code}`}
                          >
                            <QrCode className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => navigate(`/analytics/${link.id}`)}
                            data-testid={`analytics-${link.short_code}`}
                          >
                            <BarChart3 className="h-4 w-4" />
                          </Button>
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" size="icon" data-testid={`more-${link.short_code}`}>
                                <MoreVertical className="h-4 w-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem
                                onClick={() => window.open(link.target_url, '_blank')}
                              >
                                <ExternalLink className="mr-2 h-4 w-4" />
                                Visit Original
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onClick={() => handleDelete(link.id)}
                                className="text-destructive"
                              >
                                <Trash2 className="mr-2 h-4 w-4" />
                                Delete
                              </DropdownMenuItem>
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            ) : (
              <div className="text-center py-12">
                <Link2 className="h-12 w-12 mx-auto mb-4 text-muted-foreground opacity-50" />
                <h3 className="text-lg font-medium mb-2">No links yet</h3>
                <p className="text-muted-foreground mb-4">
                  Create your first short link to get started
                </p>
                <Button onClick={() => setCreateDialogOpen(true)}>
                  <Plus className="mr-2 h-4 w-4" />
                  Create Link
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        {/* QR Code Dialog */}
        <Dialog open={qrDialogOpen} onOpenChange={setQrDialogOpen}>
          <DialogContent className="max-w-sm">
            <DialogHeader>
              <DialogTitle>QR Code</DialogTitle>
              <DialogDescription>
                Scan to visit {selectedLink?.short_code}
              </DialogDescription>
            </DialogHeader>
            {selectedLink?.qr_code && (
              <div className="flex flex-col items-center gap-4">
                <img
                  src={`data:image/png;base64,${selectedLink.qr_code}`}
                  alt="QR Code"
                  className="w-48 h-48 border rounded-lg"
                />
                <Button
                  onClick={() => {
                    const a = document.createElement('a');
                    a.href = `data:image/png;base64,${selectedLink.qr_code}`;
                    a.download = `qr-${selectedLink.short_code}.png`;
                    a.click();
                  }}
                  variant="outline"
                >
                  Download QR Code
                </Button>
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  );
};

export default LinksPage;
